/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>

#include "common.h"
#include "config_wrapper.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/inputtext.h"
#include "xine-toolkit/labelbutton.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/button.h"
#include "xine-toolkit/combo.h"
#include "xine-toolkit/browser.h"
#include "xine-toolkit/window.h"
#include "file_browser.h"
#include "actions.h"
#include "event.h"
#include "errors.h"

#define WINDOW_WIDTH        500
#define WINDOW_HEIGHT       390
#define MAX_DISP_ENTRIES    10

#define MAXFILES            65535

#ifndef S_ISLNK
#define S_ISLNK(mode)  0
#endif
#ifndef S_ISFIFO
#define S_ISFIFO(mode) 0
#endif
#ifndef S_ISSOCK
#define S_ISSOCK(mode) 0
#endif
#ifndef S_ISCHR
#define S_ISCHR(mode)  0
#endif
#ifndef S_ISBLK
#define S_ISBLK(mode)  0
#endif
#ifndef S_ISREG
#define S_ISREG(mode)  0
#endif
#if !S_IXUGO
#define S_IXUGO        (S_IXUSR | S_IXGRP | S_IXOTH)
#endif

typedef struct {
  uint32_t have, used, bufsize, bufused;
  char **array, *buf;
} fb_list_t;

static void _fb_list_init (fb_list_t *list) {
  list->have = list->used = list->bufsize = list->bufused = 0;
  list->array = NULL;
  list->buf = NULL;
}

static void _fb_list_deinit (fb_list_t *list) {
  list->have = list->used = list->bufsize = list->bufused = 0;
  SAFE_FREE (list->array);
  SAFE_FREE (list->buf);
}

static void _fb_list_clear (fb_list_t *list) {
  list->used = list->bufused = 0;
}

static void _fb_list_dummy (fb_list_t *list) {
  if (!list->have) {
    list->array = malloc (512 * sizeof (list->array[0]));
    if (!list->array)
      return;
    list->have = 512;
  }
  if (!list->bufsize) {
    list->buf = malloc (8192);
    if (!list->buf)
      return;
    list->bufsize = 8192;
  }
  list->used = 1;
  list->bufused = 8;
  memcpy (list->buf, "\0\0\0\0../", 8);
  list->array[0] = list->buf + 4;
  list->array[1] = NULL;
}

static void _fb_list_add (fb_list_t *list, const char *str, uint32_t slen) {
  char *p;

  if (list->used + 2 > list->have) {
    uint32_t nhave = list->have + 512;
    char **na = realloc (list->array, nhave * sizeof (list->array[0]));
    if (!na)
      return;
    list->array = na;
    list->have = nhave;
  }
  /* keep start and end pads */
  if (list->bufused + slen + 8 > list->bufsize) {
    uint32_t nsize = list->bufsize + 8192;
    char **a, *nb = realloc (list->buf, nsize), **e;
    if (!nb)
      return;
    if (!list->buf) {
      memset (nb, 0, 4);
      list->bufused = 4;
    }
    for (a = list->array, e = list->array + list->used; a < e; a++)
      *a = nb + (*a - list->buf);
    list->buf = nb;
    list->bufsize = nsize;
  }
  p = list->buf + list->bufused;
  list->array[list->used++] = p;
  list->array[list->used] = NULL;
  memcpy (p, str, slen);
  p[slen] = 0;
  list->bufused += slen + 1;
}

static void fb_list_reverse (fb_list_t *list) {
  char **a, **b;
  if (list->used < 2)
    return;
  for (a = list->array, b = list->array + list->used - 1; a < b; a++, b--) {
    char *temp = *a;
    *a = *b;
    *b = temp;
  }
}

typedef enum {
  _W_origin = 0,
  _W_directories_browser,
  _W_files_browser,
  _W_directories_sort,
  _W_files_sort,
  /* keep order */
  _W_rename,
  _W_delete,
  _W_cb_button0,
  /* /keep order */
  _W_create,
  _W_filters,
  _W_hidden_box,
  _W_hidden_label,
  _W_close,
  _W_cb_button1,
  _W_LAST
} _W_t;

struct filebrowser_s {
  gui_new_window_t                nw;
  char                            id[16];

  xitk_window_t                  *transient_for;

  xitk_widget_t                  *w[_W_LAST];
  unsigned int                    num_need_file;

  int                             sort_direction[2]; /** << dir, file; down, up */
  int                             sort_w, sort_h;
  xitk_image_t                   *sort_skin; /** << top (down), bottom (up) */

  const char                    **file_filters;
  int                             filter_selected;

  int                             show_hidden_files;

  fb_list_t                       list[2]; /** << dir, file */
  int                             file_sel;

  filebrowser_callback_button_t   cbb[3];
  int                           (*exit) (filebrowser_t *fb, uint32_t num);

  xitk_register_key_t             dialog;

  struct {
    xitk_window_t                *xwin;
    xitk_widget_list_t           *wl;
    xitk_widget_t                *input, *yes, *no;
    void                        (*callback) (filebrowser_t *fb, const char *newname);
    xitk_register_key_t           widget_key;
  }                               fne;

  uint32_t                        saved_file_len;
  /* offsets into path */
  uint32_t                        dirstart, dirstop, filestart, filestop;
  char                            path[XITK_PATH_MAX], trypath[XITK_PATH_MAX];
  char                            saved_file[768];
};

static const char *_fb_file_get (filebrowser_t *fb, const char *name) {
  uint32_t have, l;

  if (!fb->saved_file_len) {
    have = sizeof (fb->saved_file);
    l = fb->filestop - fb->filestart + 1;
    if (l > have)
      l = have;
    fb->saved_file_len = l;
    memcpy (fb->saved_file, fb->path + fb->filestart, l);
  }

  have = sizeof (fb->path) - fb->filestart - 1;
  l = xitk_find_byte (name, 0) + 1;
  if (l > have)
    l = have;
  memcpy (fb->path + fb->filestart, name, l);
  fb->filestop = fb->filestart + l - 1;
  fb->path[fb->filestop] = 0;

  return fb->path + fb->dirstart;
}

static void _fb_file_unget (filebrowser_t *fb) {
  if (!fb->saved_file_len)
    return;
  memcpy (fb->path + fb->filestart, fb->saved_file, fb->saved_file_len);
  fb->filestop = fb->filestart + fb->saved_file_len - 1;
  fb->saved_file_len = 0;
}

/*
 * Enable/disable widgets in file browser window
 */
static void fb_deactivate(filebrowser_t *fb) {
  xitk_widgets_state (fb->w, _W_LAST, XITK_WIDGET_STATE_ENABLE, 0);
}
static void fb_reactivate(filebrowser_t *fb) {
  xitk_widgets_state (fb->w, _W_LAST, XITK_WIDGET_STATE_ENABLE, ~0u);
}

/*
 * **************************************************
 */

/*
 * ************************* filename editor **************************
 */
static int fne_destroy (filebrowser_t *fb) {
  xitk_unregister_event_handler (fb->nw.gui->xitk, &fb->fne.widget_key);
  xitk_window_destroy_window (fb->fne.xwin);
  fb->fne.xwin = NULL;
  return 1;
}

static void fne_button_cb (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;

  (void)state;
  (void)modifier;
  if ((w == fb->fne.yes) && fb->fne.callback)
    fb->fne.callback (fb, xitk_inputtext_get_text (fb->fne.input));
  fb_reactivate (fb);
  fne_destroy (fb);
}

static int fne_event (void *data, const xitk_be_event_t *e) {
  filebrowser_t *fb = data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      fb_reactivate (fb);
      return fne_destroy (fb);

    case XITK_EV_KEY_DOWN:
    case XITK_EV_KEY_UP:
      if ((e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE))
        return xitk_widget_key_event (fb->fne.no, e, 1);
      break;

    default: ;
  }
  return gui_handle_be_event (fb->nw.gui, e);
}

static void fb_create_input_window (filebrowser_t *fb, void (*cb) (filebrowser_t *fb, const char *newname),
  const char *title) {
  int x, y, w, width = WINDOW_WIDTH, height = 102;

  if (fb->fne.xwin)
    return;

  fb->fne.callback = cb;

  fb->fne.xwin = xitk_window_create_window_ext (fb->nw.gui->xitk,
    fb->nw.wr.x + ((fb->nw.wr.width - width) >> 1),
    fb->nw.wr.y + ((fb->nw.wr.height - height) >> 1),
    width, height, title,
    NULL, "xine", 0, gui_layer_above (fb->nw.gui, NULL), fb->nw.gui->icon, XITK_WINDOW_BG_FRAME);
  xitk_window_set_wm_window_type (fb->fne.xwin, WINDOW_TYPE_NORMAL);
  fb->fne.wl = xitk_window_widget_list (fb->fne.xwin);

  {
    xitk_labelbutton_widget_t lb = {
      .nw = { .wl = fb->fne.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .button_type = CLICK_BUTTON,
      .align = ALIGN_CENTER,
      .callback = fne_button_cb
    };
    xitk_inputtext_widget_t inp = {
      .nw = { .wl = fb->fne.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .text = fb->trypath,
      .max_length = sizeof (fb->trypath)
    };

    x = 15;
    y = 30;
    w = width - 30;

    fb->fne.input = xitk_noskin_inputtext_create (&inp,
      x, y, w, 20, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);

    y = height - (23 + 15);
    x = 15;
    w = 100;

    lb.label    = _("Apply");
    fb->fne.yes = xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    x = width - (w + 15);

    lb.label    = _("Cancel");
    fb->fne.no = xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);
  }

  {
    char buffer[256];
    snprintf (buffer, sizeof(buffer), "filenameed%u", (unsigned int) time(NULL));
    fb->fne.widget_key = xitk_be_register_event_handler (buffer, fb->fne.xwin, fne_event, fb, NULL, NULL);
  }

  xitk_window_set_transient_for_win (fb->fne.xwin, fb->nw.xwin);
  xitk_window_flags (fb->fne.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (fb->fne.xwin);
}
/*
 * ************************** END OF filename editor **************************
 */

/*
 * **************************** File related funcs **************************
 */

static void fb_extract_path_and_file(filebrowser_t *fb, const char *filepathname) {
  uint32_t l, have, set_homedir = 0;
  char *d, temp;

  if (filepathname && filepathname[0]) {
    ;
  } else {
    filepathname = "~";
  }

  have = sizeof (fb->path) - fb->dirstart;
  d = fb->path + fb->dirstart;

  if ((filepathname[0] == '\\') && (filepathname[1] == '~'))
    /* "\~..." -> "~..." */
    filepathname++;
  if ((filepathname[0] == '~') && ((filepathname[1] == '/') || !filepathname[1])) {
    /* "<any>" + "~/..." -> "/home/user" "/" "..." */
    filepathname++;
    set_homedir = 1;
  } else if (filepathname[0] && (filepathname[0] != '/') && (fb->filestart > fb->dirstart)) {
    /* "<unset>" + "somename" -> "/home/user" "/" "somename" */
    set_homedir = 2;
  }
  if (set_homedir) {
    const char *homedir = xine_get_homedir ();

    l = strlcpy (d, homedir, have);
    if (l > have - 2)
      l = have - 2;
    d += l;
    fb->dirstop = d - fb->path;
    have -= l;
    *d++ = '/';
    fb->filestop = fb->filestart = d - fb->path;
    have--;
    *d = 0;
  }

  if (!filepathname[0])
    return;
  if (filepathname[0] != '/') {
    /* "/some/dir/filename.flv" "newfile.flv" -> "/some/dir/newfile.flv" */
    have = sizeof (fb->path) - fb->dirstop;
    d = fb->path + fb->dirstop;
    *d++ = '/';
    have--;
  } else {
    /* "<any>" + "/////////foo/bar" -> "/foo/bar" */
    while (filepathname[1] == '/')
      filepathname++;
  }

  l = strlcpy (d, filepathname, have);
  if (l > have - 1)
    l = have - 1;
  d += l;
  /* "...////" -> "..." */
  fb->path[fb->dirstart - 1] = 0;
  temp = fb->path[fb->dirstart];
  fb->path[fb->dirstart] = 0;
  while (d[-1] == '/')
    d--;
  fb->path[fb->dirstart] = temp;
  *d = 0;

  /* file part? */
  if (xitk_filetype (fb->path + fb->dirstart) == XITK_FILETYPE_DIR) {
    /* known a dir */
    fb->dirstop = d - fb->path;
    if (d[-1] != '/') {
      *d++ = '/';
      *d = 0;
    }
    fb->filestart = fb->filestop = d - fb->path;
  } else {
    fb->filestop = d - fb->path;
    /* find last / */
    fb->path[fb->dirstart -1] = '/';
    while (d[-1] != '/')
      d--;
    fb->dirstop = fb->filestart = d - fb->path;
    if (fb->dirstop < fb->dirstart + 2) {
      /* "justafilename.flv" or "/filename.flv" */
      ;
    } else {
      /* "...somedir/filename.flv" */
      fb->dirstop--;
    }
  }
}

/*
 * Sorting function, it comes from GNU fileutils package.
 */
#define S_N        0x0
#define S_I        0x4
#define S_F        0x8
#define S_Z        0xC
#define CMP          2
#define LEN          3
#define ISDIGIT(c)   (((c) ^ '0') <= 9)
static int my_strverscmp(const char *s1, const char *s2) {
  const unsigned char *p1 = (const unsigned char *) s1;
  const unsigned char *p2 = (const unsigned char *) s2;
  unsigned char c1, c2;
  int state;
  int diff;
  static const uint8_t next_state[] = {
    S_N, S_I, S_Z, S_N,
    S_N, S_I, S_I, S_I,
    S_N, S_F, S_F, S_F,
    S_N, S_F, S_Z, S_Z
  };
  static const int8_t result_type[] = {
    CMP, CMP, CMP, CMP, CMP, LEN, CMP, CMP,
    CMP, CMP, CMP, CMP, CMP, CMP, CMP, CMP,
    CMP,  -1,  -1, CMP,   1, LEN, LEN, CMP,
      1, LEN, LEN, CMP, CMP, CMP, CMP, CMP,
    CMP, CMP, CMP, CMP, CMP, LEN, CMP, CMP,
    CMP, CMP, CMP, CMP, CMP, CMP, CMP, CMP,
    CMP,   1,   1, CMP,  -1, CMP, CMP, CMP,
     -1, CMP, CMP, CMP
  };

  if(p1 == p2)
    return 0;

  c1 = *p1++;
  c2 = *p2++;

  state = S_N | ((c1 == '0') + (ISDIGIT(c1) != 0));

  while((diff = c1 - c2) == 0 && c1 != '\0') {
    state = next_state[state];
    c1 = *p1++;
    c2 = *p2++;
    state |= (c1 == '0') + (ISDIGIT(c1) != 0);
  }

  state = result_type[state << 2 | ((c2 == '0') + (ISDIGIT(c2) != 0))];

  switch(state) {
  case CMP:
    return diff;

  case LEN:
    while(ISDIGIT(*p1++))
      if(!ISDIGIT(*p2++))
	return 1;

    return ISDIGIT(*p2) ? -1 : diff;

  default:
    return state;
  }
}

/*
 * Wrapper to my_strverscmp() for qsort() calls, which sort mrl_t type array.
 */
static int _sortfiles_default (const void *a, const void *b) {
  const char * const *d = (const char * const *)a;
  const char * const *e = (const char * const *)b;
  return my_strverscmp (*d, *e);
}

static void fb_list_sort (fb_list_t *list) {
  if (list->used < 2)
    return;
  qsort (list->array, list->used, sizeof (list->array[0]), _sortfiles_default);
}

static void sort_directories (filebrowser_t *fb) {
  fb_list_sort (&fb->list[0]);
  if (fb->sort_direction[0])
    fb_list_reverse (&fb->list[0]);
  xitk_browser_update_list (fb->w[_W_directories_browser],
    (const char *const *)fb->list[0].array, NULL, fb->list[0].used, 0);
}

static int sort_files (filebrowser_t *fb) {
  int ret = -1;

  fb_list_sort (&fb->list[1]);

  if (fb->list[1].used > 0) {
    unsigned int b = 0, e = fb->list[1].used;

    do {
      unsigned int m = (b + e) >> 1;
      int d = my_strverscmp (fb->path + fb->filestart, fb->list[1].array[m]);

      if (d == 0) {
        ret = m;
        break;
      } else if (d < 0) {
        e = m;
      } else {
        b = m + 1;
      }
    } while (b < e);
  }
  
  if (fb->sort_direction[1]) {
    fb_list_reverse (&fb->list[1]);
    if (ret >= 0)
      ret = fb->list[1].used - ret - 1;
  }
  xitk_browser_update_list (fb->w[_W_files_browser],
    (const char *const *)fb->list[1].array, NULL, fb->list[1].used, 0);

  return ret;
}

static void fb_getdir(filebrowser_t *fb) {
  char           temp;
  struct dirent *pdirent;
  DIR           *pdir;

  xitk_window_define_window_cursor (fb->nw.xwin, xitk_cursor_watch);

  _fb_list_clear (&fb->list[1]);
  _fb_list_clear (&fb->list[0]);

  /* Following code relies on the fact that fb->current_dir has no trailing '/' */
  temp = fb->path[fb->dirstop];
  fb->path[fb->dirstop] = 0;
  if ((pdir = opendir (fb->path + fb->dirstart)) == NULL) {
    gui_msg (fb->nw.gui, XUI_MSG_ERROR, _("Unable to open directory '%s': %s."),
      fb->path + fb->dirstart, strerror (errno));
    fb->path[fb->dirstop] = temp;
    /* give user an exit */
    _fb_list_dummy (&fb->list[0]);
    xitk_window_define_window_cursor (fb->nw.xwin, xitk_cursor_default);
    return;
  }
  fb->path[fb->dirstop] = temp;

  while ((pdirent = readdir (pdir)) != NULL) {
    fb_list_t *list;
    const char *fullname;

    /* if user don't want to see hidden files, ignore them */
    if ((pdirent->d_name[0] == '.') && pdirent->d_name[1] && (pdirent->d_name[1] != '.') && !fb->show_hidden_files)
      continue;

    fullname = _fb_file_get (fb, pdirent->d_name);
    if (xitk_filetype (fullname) == XITK_FILETYPE_DIR) {
      list = &fb->list[0];
      fb->path[fb->filestop++] = '/';
      fb->path[fb->filestop] = 0;
    } else {
      if (!filebrowser_ext_match (fb->nw.gui, pdirent->d_name, fb->filter_selected))
        continue;
      list = &fb->list[1];
    }
    _fb_list_add (list, fb->path + fb->filestart, fb->filestop - fb->filestart);
  }
  _fb_file_unget (fb);

  closedir(pdir);

  /*
   * Sort arrays
   */
  sort_directories(fb);
  {
    int i = sort_files (fb);
    if (i >= 0)
      fb->file_sel = i = xitk_browser_set_select (fb->w[_W_files_browser], i);
    xitk_widgets_state (fb->w + _W_rename, fb->num_need_file, XITK_WIDGET_STATE_ENABLE, (i >= 0) ? ~0u : 0);
  }
  xitk_window_define_window_cursor (fb->nw.xwin, xitk_cursor_default);
}

/*
 * ****************************** END OF file related funcs ***************************
 */

/*
 * ***************************** widget callbacks *******************************
 */
static void fb_select(xitk_widget_t *w, void *data, int selected, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;

  (void)modifier;
  if (w == fb->w[_W_files_browser]) {
    fb->file_sel = selected;
    xitk_widgets_state (fb->w + _W_rename, fb->num_need_file, XITK_WIDGET_STATE_ENABLE, (selected >= 0) ? ~0u : 0);
    if (selected >= 0) {
      uint32_t have = sizeof (fb->path) - fb->filestart;
      uint32_t l = strlcpy (fb->path + fb->filestart, fb->list[1].array[selected], have);

      if (l > have - 1)
        l = have - 1;
      fb->filestop = fb->filestart + l;
      xitk_inputtext_change_text (fb->w[_W_origin], fb->path + fb->dirstart);
    }
  }
}

static int _fb_exit (filebrowser_t *fb, uint32_t num) {
  filebrowser_callback_button_t *b = fb->cbb + num;

  if (b->need_a_file && !fb->path[fb->filestart])
    return 0;
  /* user info */
  if (b->callback)
    b->callback (fb, b->userdata);
  /* close */
  if (fb->fne.xwin)
    fne_destroy (fb);
  if (fb->dialog)
    xitk_unregister_event_handler (fb->nw.gui->xitk, &fb->dialog);

  gui_window_delete (&fb->nw);

  filebrowser_ext_list_unget (fb->nw.gui, &fb->file_filters);

  xitk_image_free_image (&fb->sort_skin);

  _fb_list_deinit (&fb->list[1]);
  _fb_list_deinit (&fb->list[0]);

  free (fb);
  return 1;
}

static void _fb_button (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *)data;

  (void)state;
  (void)modifier;
  fb->exit (fb, (w == fb->w[_W_cb_button0]) ? 0 : (w == fb->w[_W_cb_button1]) ? 2 : 1);
}

static void fb_dbl_select(xitk_widget_t *w, void *data, int selected, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;

  (void)modifier;
  if (w == fb->w[_W_directories_browser]) {
    uint32_t l = xitk_find_byte (fb->list[0].array[selected], 0);

    if (!memcmp (fb->list[0].array[selected], "./", 3)) {
      /* Want to re-read current dir */
      ;
    } else if (!memcmp (fb->list[0].array[selected], "../", 4)) {
      char *d, *f;
      /* level up */
      fb->path[fb->dirstart - 1] = '/';
      d = fb->path + fb->dirstop;
      while (d[-1] != '/')
        d--;
      fb->dirstop = d - fb->path;
      if (fb->dirstop < fb->dirstart + 2) {
        /* all dirs gone or leading / reached */
        ;
      } else {
        fb->dirstop--;
      }
      f = fb->path + fb->filestart;
      if (f > d) {
        memmove (d, f, fb->filestop - fb->filestart + 1);
        fb->filestart -= f - d;
        fb->filestop -= f - d;
      }
    } else if (fb->filestop + l < sizeof (fb->path)) {
      char *d = fb->path + fb->dirstop, *f = fb->path + fb->filestart;
      /* level down */
      fb->path[fb->dirstart - 1] = '/';
      if (d[-1] == '/') {
        /* "file" -> "add/file" or "/" -> "/add/file" */
        ;
      } else {
        /* "...have/file" -> "...have/add/file" */
        d++;
      }
      memmove (d + l, f, fb->filestop - fb->filestart + 1);
      memcpy (d, fb->list[0].array[selected], l);
      d += l;
      fb->dirstop = d - fb->path - 1;
      l = d - f;
      fb->filestart += l;
      fb->filestop += l;
    }
    xitk_inputtext_change_text (fb->w[_W_origin], fb->path + fb->dirstart);
    fb_getdir(fb);
  } else if (w == fb->w[_W_files_browser]) {
    uint32_t l = xitk_find_byte (fb->list[1].array[selected], 0);

    fb->file_sel = selected;
    xitk_widgets_state (fb->w + _W_rename, 2, XITK_WIDGET_STATE_ENABLE, ~0u);
    if (fb->filestop + l < sizeof (fb->path)) {
      memcpy (fb->path + fb->filestart, fb->list[1].array[selected], l + 1);
      fb->filestop = fb->filestart + l;
      fb->exit (fb, 0);
    }
  }
}

static void fb_change_origin(xitk_widget_t *w, void *data, const char *currenttext) {
  filebrowser_t *fb = (filebrowser_t *)data;

  (void)w;
  fb_extract_path_and_file(fb, currenttext);
  xitk_inputtext_change_text (fb->w[_W_origin], fb->path + fb->dirstart);
  fb_getdir(fb);
}

static void fb_sort(xitk_widget_t *w, void *data) {
  xitk_widget_t *_w = w;
  filebrowser_t *fb = (filebrowser_t *) data;
  uint32_t mode = (w == fb->w[_W_directories_sort]) ? 0 : 1;

  fb->sort_direction[mode] ^= 1;

  xitk_widgets_state (&_w, 1, XITK_WIDGET_STATE_VISIBLE, 0);
  xitk_image_copy_rect (fb->sort_skin, xitk_get_widget_foreground_skin (w),
    0, fb->sort_direction[mode] * fb->sort_h,
    fb->sort_w, fb->sort_h, 0, 0);
  xitk_widgets_state (&_w, 1, XITK_WIDGET_STATE_VISIBLE, ~0u);

  fb_list_reverse (&fb->list[mode]);
  xitk_browser_update_list (fb->w[_W_directories_browser + mode],
    (const char * const *)fb->list[mode].array, NULL, fb->list[mode].used, 0);

  if (mode) {
    if (fb->file_sel >= 0)
      xitk_browser_set_select (fb->w[_W_files_browser], fb->file_sel = fb->list[1].used - fb->file_sel - 1);
  }
}

static void fb_rm_2 (void *data, int state) {
  filebrowser_t *fb = data;

  if (state == 2) {
    const char *fullname = _fb_file_get (fb, fb->list[1].array[fb->file_sel]);
    int r = unlink (fullname);

    if (r == -1)
      gui_msg (fb->nw.gui, XUI_MSG_ERROR, _("Unable to delete file '%s': %s."), fullname, strerror (errno));
    _fb_file_unget (fb);
    if (r != -1) {
      fb_getdir (fb);
      xitk_widgets_state (fb->w + _W_rename, fb->num_need_file, XITK_WIDGET_STATE_ENABLE, (fb->file_sel >= 0) ? ~0u : 0);
    }
  }
  fb_reactivate (fb);
}

static void fb_rename_2 (filebrowser_t *fb, const char *newname) {
  const char *fullname = _fb_file_get (fb, fb->list[1].array[fb->file_sel]);
  int r = rename (fullname, newname);

  if (r == -1)
    gui_msg (fb->nw.gui, XUI_MSG_ERROR, _("Unable to rename file '%s' to '%s': %s."), fullname, newname, strerror (errno));
  _fb_file_unget (fb);
  if (r != -1)
    fb_getdir (fb);
  fb_reactivate (fb);
}

static void fb_rename_delete (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;

  (void)state;
  (void)modifier;
  if (fb->file_sel >= 0) {
    const char *fullname;

    fb_deactivate (fb);
    fullname = _fb_file_get (fb, fb->list[1].array[fb->file_sel]);
    if (w == fb->w[_W_delete]) {
      fb->dialog = xitk_window_dialog_3 (fb->nw.gui->xitk, fb->nw.xwin,
        gui_layer_above (fb->nw.gui, NULL), 400, _("Confirm deletion ?"), fb_rm_2, fb,
        NULL, XITK_LABEL_YES, XITK_LABEL_NO, NULL, 0, ALIGN_DEFAULT, "%s", fullname);
    } else {
      strlcpy (fb->trypath, fullname, sizeof (fb->trypath));
      fb_create_input_window (fb, fb_rename_2, _("Rename file"));
    }
    _fb_file_unget (fb);
  }
}

static void fb_mkdir_2 (filebrowser_t *fb, const char *newdir) {
  int r = mkdir_safe (newdir);

  if (r)
    gui_msg (fb->nw.gui, XUI_MSG_ERROR, _("Unable to create the directory '%s': %s."), newdir, strerror (r));
  else
    fb_getdir (fb);
  fb_reactivate (fb);
}

static void fb_mkdir_1 (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;
  uint32_t       l = fb->filestart - fb->dirstart;

  (void)w;
  (void)state;
  (void)modifier;

  memcpy (fb->trypath, fb->path + fb->dirstart, l);
  fb->trypath[l] = 0;
  fb_deactivate (fb);
  fb_create_input_window (fb, fb_mkdir_2, _("Create a new directory"));
}

static void fb_select_filter (xitk_widget_t *w, void *data, int selected, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *) data;

  (void)w;
  (void)modifier;
  fb->filter_selected = selected;
  fb_getdir(fb);
}

static void fb_hidden_files (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  filebrowser_t *fb = (filebrowser_t *)data;

  (void)w;
  (void)modifier;
  fb->show_hidden_files = state;
  config_update_bool (fb->nw.gui->xine, "media.files.show_hidden_files", state);
  fb_getdir (fb);
}

static int fb_event (void *data, const xitk_be_event_t *e) {
  filebrowser_t *fb = (filebrowser_t *)data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      if (xitk_is_widget_enabled (fb->w[_W_close])) /* Exit only if close button would exit */
        return fb->exit (fb, 1);
      break;

    case XITK_EV_POS_SIZE:
      fb->nw.wr.x = e->x;
      fb->nw.wr.y = e->y;
      fb->nw.wr.width = e->w;
      fb->nw.wr.height = e->h;
      return 0;

    case XITK_EV_KEY_DOWN:
    case XITK_EV_KEY_UP:
      if ((e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE))
        return xitk_widget_key_event (fb->w[_W_close], e, 1);
      break;

    default: ;
  }

  return gui_handle_be_event (fb->nw.gui, e);
}

void filebrowser_raise_window(filebrowser_t *fb) {
  if (fb)
    gui_raise_window (fb->nw.gui, fb->nw.xwin);
}

void filebrowser_end (filebrowser_t *fb) {
  if (fb)
    fb->exit (fb, 1);
}

size_t filebrowser_get_string (filebrowser_t *fb, char *buf, size_t blen, int dir0_fullname1) {
  size_t l;
  if (!fb || !buf)
    return 0;
  l = (dir0_fullname1 == 0) ? fb->dirstop - fb->dirstart : fb->filestop- fb->dirstart;
  if (!l)
    return 0;
  if (l > blen - 1)
    l = blen - 1;
  memcpy (buf, fb->path + fb->dirstart, l);
  buf[l] = 0;
  return l;
}

const char * const *filebrowser_get_all_files (filebrowser_t *fb) {
  return (const char * const *)fb->list[1].array;
}

filebrowser_t *filebrowser_create (gGui_t *gui, xitk_window_t *transient_for,
  const char *window_title, const char *filepathname,
  const filebrowser_callback_button_t *ok_close_extra, unsigned int num_buttons, int initial_filter) {
  filebrowser_t *fb;
  int            x, y, w;

  if (!gui)
    return NULL;

  fb = (filebrowser_t *)calloc (1, sizeof (*fb));
  if (!fb)
    return NULL;

  memcpy (fb->id, "filebrowser_0", 13);
  fb->id[12] += initial_filter;
  fb->nw.gui = gui;
  fb->nw.id = fb->id;
  fb->nw.title = window_title ? window_title : _("File Browser");
  fb->nw.skin = NULL;
  fb->nw.wfskin = NULL;
  fb->nw.adjust = NULL;
  fb->nw.wr.x = XITK_WINDOW_POS_CENTER;
  fb->nw.wr.y = XITK_WINDOW_POS_CENTER;
  fb->nw.wr.width = WINDOW_WIDTH;
  fb->nw.wr.height = WINDOW_HEIGHT;

  fb->fne.xwin = NULL;
  fb->exit = _fb_exit;

  _fb_list_init (&fb->list[0]);
  _fb_list_init (&fb->list[1]);
  fb->file_sel = -1;

  fb->dirstart = fb->dirstop = fb->filestart = fb->filestop = 4;
  fb->path[4] = 0;

  if (num_buttons > 3)
    num_buttons = 3;
  fb->cbb[0].label = NULL;
  fb->cbb[1].label = NULL;
  fb->cbb[2].label = NULL;
  {
    unsigned int u;
    for (u = 0; u < num_buttons; u++)
      fb->cbb[u] = ok_close_extra[u];
  }
  fb->cbb[1].need_a_file = 0;
  fb->num_need_file = fb->cbb[0].need_a_file ? 3 : 2;

  /* Create window */
  gui_window_new (&fb->nw);

  fb->sort_direction[0] = 0;
  fb->sort_direction[1] = 0;
  {
    xine_cfg_entry_t  cfg_entry;

    if (xine_config_lookup_entry (fb->nw.gui->xine, "media.files.show_hidden_files", &cfg_entry))
      fb->show_hidden_files = cfg_entry.num_value;
  }

  fb_extract_path_and_file(fb, filepathname);

  fb->file_filters = filebrowser_ext_list_get (fb->nw.gui);
  fb->filter_selected = initial_filter;

  {
    xitk_browser_widget_t br = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .browser = { .max_displayed_entries = MAX_DISP_ENTRIES },
      .callback = fb_select,
      .dbl_click_callback = fb_dbl_select
    };
    xitk_inputtext_widget_t inp = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .max_length = XITK_PATH_MAX + XITK_NAME_MAX + 1,
      .text = fb->path + fb->dirstart,
      .callback = fb_change_origin
    };
    xitk_button_widget_t b = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .symbol = XITK_SYMBOL_USER,
      .callback = fb_sort
    };

    x = 15;
    y = 30;
    w = WINDOW_WIDTH - 30;

    fb->w[_W_origin] = xitk_noskin_inputtext_create (&inp,
      x, y, w, 20, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, fontname);

    y += 45;
    w = (WINDOW_WIDTH - 30 - 10) / 2;

    br.browser.num_entries           = fb->list[0].used;
    br.browser.entries               = (const char *const *)fb->list[0].array;
    fb->w[_W_directories_browser] = xitk_noskin_browser_create (&br,
      x + 2, y + 2, w - 4 - 12, 20, 12, fontname);

    xitk_image_draw_rectangular_box (fb->nw.bg, x, y, w, xitk_get_widget_height (fb->w[_W_directories_browser]) + 4, XITK_DRAW_INNER);

    y -= 15;

    fb->w[_W_directories_sort] = xitk_noskin_button_create (&b, x, y, w, 15);

    x = WINDOW_WIDTH - (w + 15);
    y += 15;

    br.browser.num_entries = fb->list[1].used;
    br.browser.entries     = (const char * const *)fb->list[1].array;
    fb->w[_W_files_browser] = xitk_noskin_browser_create (&br,
      x + 2, y + 2, w - 4 - 12, 20, 12, fontname);

    xitk_image_draw_rectangular_box (fb->nw.bg, x, y, w, xitk_get_widget_height (fb->w[_W_files_browser]) + 4, XITK_DRAW_INNER);

    y -= 15;

    fb->w[_W_files_sort] = xitk_noskin_button_create (&b, x, y, w, 15);
  }

  {
    xitk_image_t *dsimage = xitk_get_widget_foreground_skin(fb->w[_W_directories_sort]);
    xitk_image_t *fsimage = xitk_get_widget_foreground_skin(fb->w[_W_files_sort]);
    int           i, j, w, dy;
    int color[4] = {
      xitk_get_cfg_num (fb->nw.gui->xitk, XITK_BLACK_COLOR),
      xitk_get_cfg_num (fb->nw.gui->xitk, XITK_SELECT_COLOR),
      xitk_get_cfg_num (fb->nw.gui->xitk, XITK_WHITE_COLOR),
      xitk_get_cfg_num (fb->nw.gui->xitk, XITK_FOCUS_COLOR)
    };
    xitk_part_image_t pi;
    xitk_point_t points[4];

    pi.x = pi.y = 0;
    pi.width = fb->sort_w = xitk_image_width (dsimage);
    pi.height = fb->sort_h = xitk_image_height (dsimage);
    pi.image = fb->sort_skin = xitk_image_new (fb->nw.gui->xitk, NULL, 0, fb->sort_w, fb->sort_h * 2);
    pi.num_states = 3;

    xitk_part_image_draw_bevel_style (&pi, XITK_DRAW_BEVEL);
    w = fb->sort_w / 3;
    for (i = 0; i < 4 * 2; i += 4) {
      xitk_image_draw_rectangular_box (fb->sort_skin,          5,     4     + i, w - 44, 2, XITK_DRAW_OUTTER);
      xitk_image_draw_rectangular_box (fb->sort_skin, w     - 20,     4     + i, 11,     2, XITK_DRAW_OUTTER);
      xitk_image_draw_rectangular_box (fb->sort_skin, w     +  5,     4     + i, w - 44, 2, XITK_DRAW_OUTTER);
      xitk_image_draw_rectangular_box (fb->sort_skin, w * 2 - 20,     4     + i, 11,     2, XITK_DRAW_OUTTER);
      xitk_image_draw_rectangular_box (fb->sort_skin, w * 2 +  5 + 1, 4 + 1 + i, w - 44, 2, XITK_DRAW_OUTTER);
      xitk_image_draw_rectangular_box (fb->sort_skin, w * 3 - 20 + 1, 4 + 1 + i, 11 + 1, 2, XITK_DRAW_OUTTER);
    }
    xitk_image_copy_rect (fb->sort_skin, fb->sort_skin, 0, 0, fb->sort_w, fb->sort_h, 0, fb->sort_h);

    points[0].x = w - 30;      points[0].y = 2 + 8;
    points[1].x = w - 30 - 6;  points[1].y = 3;
    points[2].x = w - 30 + 6;  points[2].y = 3;
    points[3].x = points[0].x; points[3].y = points[0].y;
    dy = fb->sort_h + 1;
    j = 0;
    while (1) {
      xitk_image_fill_polygon (fb->sort_skin, points, 4, color[3]);
      for (i = 0; i < 3; i++)
        xitk_image_draw_line (fb->sort_skin, points[i].x, points[i].y, points[i + 1].x, points[i + 1].y, color[i]);
      for (i = 0; i < 4; i++)
        points[i].x += w;
      xitk_image_fill_polygon (fb->sort_skin, points, 4, color[3]);
      for (i = 0; i < 3; i++)
        xitk_image_draw_line (fb->sort_skin, points[i].x, points[i].y, points[i + 1].x, points[i + 1].y, color[i]);
      for (i = 0; i < 4; i++)
        points[i].x += w + 1, points[i].y += dy;
      xitk_image_fill_polygon (fb->sort_skin, points, 4, color[3]);
      for (i = 0; i < 3; i++)
        xitk_image_draw_line (fb->sort_skin, points[i].x, points[i].y, points[i + 1].x, points[i + 1].y, color[i]);
      if (j)
        break;
      points[0].x = w - 30;      points[0].y = 2     + fb->sort_h;
      points[1].x = w - 30 - 7;  points[1].y = 2 + 8 + fb->sort_h;
      points[2].x = w - 30 + 7;  points[2].y = 2 + 8 + fb->sort_h;
      points[3].x = points[0].x; points[3].y = points[0].y;
      dy = 1 - fb->sort_h;
      j = 1;
    }

    xitk_image_copy_rect (fb->sort_skin, dsimage, 0, 0, fb->sort_w, fb->sort_h, 0, 0);
    xitk_image_copy_rect (fb->sort_skin, fsimage, 0, 0, fb->sort_w, fb->sort_h, 0, 0);
  }

  y += xitk_get_widget_height(fb->w[_W_files_browser]) + 15 + 4 + 5;

  {
    uint32_t style = XITK_DRAW_SAT (fb->nw.gui->gfx_saturation);
    xitk_labelbutton_widget_t lb = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .button_type = CLICK_BUTTON,
      .align = ALIGN_CENTER
    };
    xitk_label_widget_t lbl = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .label = _("Show hidden file")
    };
    xitk_combo_widget_t cmb = {
      .nw = { .wl = fb->nw.wl, .userdata = fb, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE },
      .layer_above = gui_layer_above (fb->nw.gui, NULL),
      .select = fb->filter_selected,
      .parent_wkey = &fb->nw.key,
      .entries  = fb->file_filters,
      .callback = fb_select_filter
    };
    xitk_button_widget_t b = {
      .nw = {
        .wl = fb->nw.wl,
        .userdata = fb,
        .add_state = fb->show_hidden_files
                   ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                   : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE)
      },
      .symbol = XITK_SYMBOL_CHECK,
      .state_callback = fb_hidden_files
    };

    fb->w[_W_filters] = xitk_noskin_combo_create (&cmb, x, y, w, 18);

    x = 15;

    fb->w[_W_hidden_box] = xitk_noskin_button_create (&b, x, y + 3, 14, 14);

    fb->w[_W_hidden_label] = xitk_noskin_label_create (&lbl, x + 17, y, w - 15, 20, fontname);
    xitk_widget_set_focus_redirect (fb->w[_W_hidden_label], fb->w[_W_hidden_box]);

    y = WINDOW_HEIGHT - (23 + 15) - (23 + 8);
    w = (WINDOW_WIDTH - (4 * 15)) / 3;

    lb.callback = fb_rename_delete;

    lb.style    = XITK_DRAW_B | style;
    lb.label    = _("Rename");
    fb->w[_W_rename] = xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    x = (WINDOW_WIDTH - w) / 2;

    lb.label    = _("Delete");
    fb->w[_W_delete] = xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    x = WINDOW_WIDTH - (w + 15);

    lb.label    = _("Create a directory");
    lb.callback = fb_mkdir_1;
    fb->w[_W_create] = xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    fb->w[_W_cb_button0] = fb->w[_W_cb_button1] = NULL;

    y = WINDOW_HEIGHT - (23 + 15);

    lb.callback = _fb_button;
    if (fb->cbb[0].label) {
      x = 15;

      lb.style    = XITK_DRAW_G | style;
      lb.label    = fb->cbb[0].label;
      fb->w[_W_cb_button0] = xitk_noskin_labelbutton_create (&lb,
        x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

      if (fb->cbb[2].label) {
        x = (WINDOW_WIDTH - w) / 2;

        lb.style    = XITK_DRAW_R | XITK_DRAW_G | style;
        lb.label    = fb->cbb[2].label;
        fb->w[_W_cb_button1] = xitk_noskin_labelbutton_create (&lb,
          x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);
      }
    }

    x = WINDOW_WIDTH - (w + 15);

    lb.style    = XITK_DRAW_R | style;
    lb.label    = fb->cbb[1].label ? fb->cbb[1].label : _("Close");
    fb->w[_W_close] =  xitk_noskin_labelbutton_create (&lb,
      x, y, w, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);
  }

  xitk_window_set_background_image (fb->nw.xwin, fb->nw.bg);

  if(fb->w[_W_cb_button0])
    xitk_set_focus_to_widget(fb->w[_W_cb_button0]);

  {
    char buffer[256];
    snprintf(buffer, sizeof(buffer), "filebrowser%u", (unsigned int) time(NULL));
    fb->nw.key = xitk_be_register_event_handler (buffer, fb->nw.xwin, fb_event, fb, NULL, NULL);
  }
  fb->dialog = 0;

  fb->transient_for = transient_for;
  xitk_window_set_transient_for_win (fb->nw.xwin, transient_for);
  xitk_window_flags (fb->nw.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  gui_raise_window (fb->nw.gui, fb->nw.xwin);

  fb_getdir(fb);
  return fb;
}

