//===========================================================================
//
// File: partition_test.cpp
//
// Created: Mon Sep  7 12:14:13 2009
//
// Author(s): Atgeirr F Rasmussen <atgeirr@sintef.no>
//            Brd Skaflestad     <bard.skaflestad@sintef.no>
//
// $Date$
//
// $Revision$
//
//===========================================================================

/*
  Copyright 2009, 2010 SINTEF ICT, Applied Mathematics.
  Copyright 2009, 2010 Statoil ASA.

  This file is part of The Open Porous Media project  (OPM).

  OPM is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  OPM is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OPM.  If not, see <http://www.gnu.org/licenses/>.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif
#include <dune/common/mpihelper.hh>
#include <dune/grid/common/GridPartitioning.hpp>
#include <opm/core/utility/parameters/ParameterGroup.hpp>
#include <dune/grid/io/file/vtk/vtkwriter.hh>
#include <dune/grid/CpGrid.hpp>

#include <cstdlib>
#include <iterator>
#include <iostream>

using namespace Dune;

int testSinglePartition(const array<int, 3>& dims,
                        const std::array<int, 3>& split,
                        const int num_part,
                        const int num_cells,
                        const int* correct_part)
{
    // Make grid.
    CpGrid g;
    array<double, 3> sizes = {{ 1.0, 1.0, 1.0 }};
    g.createCartesian(dims, sizes);
    // Partition.
    int my_num_part = -1;
    std::vector<int> cell_part;
    partition(g, split, my_num_part, cell_part);
    std::vector<std::set<int> > overlap;
    addOverlapLayer(g, cell_part, overlap, my_num_part, true);

    // Check.
    if (my_num_part != num_part || int(cell_part.size()) != num_cells) {
        return EXIT_FAILURE;
    }
    if (!std::equal(cell_part.begin(), cell_part.end(), correct_part)) {
        return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}



int testPartitions()
{
    // Small case.
    {
        array<int, 3> dims = {{ 2, 3, 4 }};
        std::array<int, 3> split = {{ 2, 2, 2 }};
        int num_part = 8;
        const int num_cells = 24;
        int correct_part[num_cells] = { 0, 0, 4, 4, 1, 1, 5, 5,
                                        0, 0, 4, 4, 1, 1, 5, 5,
                                        2, 2, 6, 6, 3, 3, 7, 7 };
        int retval = testSinglePartition(dims, split, num_part, num_cells, correct_part);
        if (retval == EXIT_FAILURE) return EXIT_FAILURE;
    }
    // Larger case. The correct partitioning has been generated by this program,
    // so we really just test that we reproduce the same results.
    {
        array<int, 3> dims = {{ 10, 10, 10 }};
        std::array<int, 3> split = {{ 2, 3, 7 }};
        int num_part = 42;
        const int num_cells = 1000;
         int correct_part[num_cells] = {
            0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12,
            12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36,
            1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13,
            13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37,
            0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12,
            12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36,
            1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13,
            13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37,
            0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12,
            12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36,
            1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13,
            13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37,
            0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12,
            12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36, 0, 0, 6, 6, 12, 12, 18, 24, 30, 36,
            1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13,
            13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37, 1, 1, 7, 7, 13, 13, 19, 25, 31, 37,
            2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14,
            14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38,
            3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15,
            15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39,
            2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14,
            14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38,
            3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15,
            15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39,
            2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14,
            14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38, 2, 2, 8, 8, 14, 14, 20, 26, 32, 38,
            3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15,
            15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39, 3, 3, 9, 9, 15, 15, 21, 27, 33, 39,
            4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16,
            16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40,
            5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17,
            17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41,
            4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16,
            16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40,
            5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17,
            17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41,
            4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16,
            16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40, 4, 4, 10, 10, 16, 16, 22, 28, 34, 40,
            5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17,
            17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41, 5, 5, 11, 11, 17, 17, 23, 29, 35, 41
        };

         int retval = testSinglePartition(dims, split, num_part, num_cells, correct_part);
         if (retval == EXIT_FAILURE) return EXIT_FAILURE;
    }
    return EXIT_SUCCESS;
}


int main(int argc, char** argv)
{
    Dune::MPIHelper::instance(argc, argv);
    if (argc == 1) {
        // Running in test mode.
        return testPartitions();
    }

    // Running in normal mode. Make grid.
    Opm::parameter::ParameterGroup param(argc, argv);
    CpGrid grid;
    grid.init(param);

    // Partition.
    std::array<int, 3> split = {{ param.getDefault("sx", 1),
                                    param.getDefault("sy", 1),
                                    param.getDefault("sz", 1) }};
    int num_part = -1;
    std::vector<int> cell_partition;
    partition(grid, split, num_part, cell_partition, param.getDefault("recursive", false));
    std::cout << "Grid with " << cell_partition.size()
              << " cells was split into " << num_part
              << " partitions." << std::endl;

    // Output.
    VTKWriter<CpGrid::LeafGridView> vtkwriter(grid.leafView());
    vtkwriter.addCellData(cell_partition, "partition");
    std::string fname = param.get<std::string>("filename");
    std::string pname = fname.substr(0, fname.size() - 7)  + "-partition";
    vtkwriter.write(pname , VTKOptions::ascii);

    pname += ".dat";
    std::ofstream outfile(pname.c_str());
    for (int i = 0; i < int(cell_partition.size()); ++i) {
        outfile << cell_partition[i] << ' ';
        if (i%25 == 24) outfile << '\n';
    }
}
